/* batch_Homerun_H123.jsx — Updates fill+stroke colors and per-run HomePlate swap
 * - Swaps layer "HomePlateRunNum" to AE_HOMEPLATE_PATH ONCE per run (Python passes it).
 * - Colors HOMERUN text and inner precomp layers (HOME/RUN) to smart primary.
 */

(function () {
  // --- load common helpers (force reload) ---
  (function(){
    var COMMON = $.getenv("AE_COMMON_JSX") || "";
    if (COMMON) { try { delete $.global.GL; } catch(e) { $.global.GL = undefined; } $.evalFile(File(COMMON)); }
    if (!$.global.GL) throw new Error("Common library not loaded. Set AE_COMMON_JSX to gl_common.jsxinc");
  })();
  var GL = $.global.GL;

  // --- env & defaults ---
  var PROJECT     = GL.env("AE_PROJECT", null);
  var CSV_PATH    = GL.env("AE_CSV", null);
  var COMP_NAME   = GL.env("AE_COMP","Shatter-HR-192");

  var LAYER_HOMERUN    = GL.cleanValue(GL.env("AE_LAYER_HOMERUN","HOMERUN"));
  var LAYER_HR_PRECOMP = GL.cleanValue(GL.env("AE_LAYER_HR_PRECOMP","RunScored PreComp"));
  var LAYER_LOGO       = GL.cleanValue(GL.env("AE_LAYER_LOGO","TeamLogo"));
  var LAYER_TEAMNAME   = GL.cleanValue(GL.env("AE_LAYER_TEAMNAME","TeamName"));

  var LAYER_HOMEPLATE  = GL.cleanValue(GL.env("AE_LAYER_HOMEPLATE","HomePlateRunNum"));
  var HOMEPLATE_PATH   = GL.env("AE_HOMEPLATE_PATH","");

  var INNER_STROKE_NAMES = (GL.env("AE_INNER_STROKE_NAMES","TOUCH,DOWN,HOME,RUN")||"").split(",");

  var STROKE_W_STR = GL.env("AE_TEXT_STROKE_WIDTH","0.5");
  var STROKE_W     = (STROKE_W_STR!=="" && !isNaN(parseFloat(STROKE_W_STR))) ? parseFloat(STROKE_W_STR) : null;

  var OUTDIR   = GL.env("AE_OUTDIR","");
  var PATH_TPL = GL.env("AE_PATH_TEMPLATE","{league}");
  var ANIM_NAME= GL.env("AE_ANIM","H-1_2_3");
  var RS_TPL   = GL.env("AE_RS_TEMPLATE","Best Settings");
  var OM_TPL   = GL.env("AE_OM_TEMPLATE","PNG Sequence");

  var LEAGUE   = GL.env("AE_LEAGUE","");
  var LIMIT_STR= GL.env("LIMIT","");
  var LIMIT    = (LIMIT_STR && !isNaN(parseInt(LIMIT_STR,10))) ? parseInt(LIMIT_STR,10) : null;
  var PURGE    = (GL.env("AE_PURGE_BEFORE_RENDER","1")==="1");
  var NO_RENDER= (GL.env("AE_NO_RENDER","0")==="1");
  var QUIT_APP = (GL.env("AE_QUIT","1")==="1");

  var logoOpts = { dir:GL.env("AE_LOGO_DIR",""),
                   tpl:GL.env("AE_LOGO_PATH_TEMPLATE","{league}/{abbr}"),
                   exts:GL.env("AE_LOGO_EXTS","png,jpg,jpeg,svg,ai,psd") };

  // --- open project & csv (avoid reopening same project) ---
  if(!PROJECT) GL.fail("AE_PROJECT env not set.");
  var aep = new File(PROJECT); if(!aep.exists) GL.fail("AE_PROJECT not found: "+PROJECT);
  try{
    var already = app.project && app.project.file && (app.project.file.fsName === aep.fsName);
    if (!already) app.open(aep);
  }catch(e){ app.open(aep); }

  if(!CSV_PATH) GL.fail("AE_CSV env not set.");
  if(!LEAGUE || GL.cleanValue(LEAGUE)==="") GL.fail("AE_LEAGUE is required.");

  var rows   = GL.parseCSV(GL.openRead(CSV_PATH));
  var teams  = GL.buildTeams(rows, LEAGUE);
  var todo   = GL.pickTeamsLeagueOnly(teams, LEAGUE);
  if (LIMIT && todo.length>LIMIT) todo = todo.slice(0, LIMIT);
  if(!todo.length) GL.fail("No teams matched league: "+LEAGUE);

  var comp = GL.findComp(COMP_NAME);
  if(!comp) GL.fail("Comp not found: "+COMP_NAME);

  var rootOut = OUTDIR ? new Folder(OUTDIR) : (app.project.file ? app.project.file.parent : Folder.desktop);
  GL.ensureFolder(rootOut);

  if (app.beginSuppressDialogs){ try{ app.beginSuppressDialogs(); }catch(e){} }
  app.beginUndoGroup("Homerun H-RunNum - PNG Seq");

  // --- helpers ---
  function _safe01(c){ return GL.safeColor(GL.ensureNonBlack(c)); }
  function _normLoose(s){ return String(GL.cleanValue(s||"")).toLowerCase().replace(/[^a-z0-9]/g,""); }

  // remove text-animator props that override color/width
  function _killTextColorAnimators(textLayer){
    try{
      var TP = textLayer.property("ADBE Text Properties");
      var ANIMS = TP && TP.property("ADBE Text Animators");
      if (!ANIMS) return 0;
      var removed=0;
      for (var ai=ANIMS.numProperties; ai>=1; ai--){
        var A = ANIMS.property(ai);
        var PROPS = A && A.property("ADBE Text Animator Properties");
        if (!PROPS) continue;
        var hasOverride = PROPS.property("ADBE Text Fill Color") || PROPS.property("ADBE Text Stroke Color") || PROPS.property("ADBE Text Stroke Width");
        if (hasOverride){ A.remove(); removed++; }
      }
      return removed;
    }catch(e){ return 0; }
  }

  // Effects & Layer Styles sync
  function _syncEffects(layer, color01){
    var fx = layer.property("Effects"); if (!fx) return;
    for (var i=1;i<=fx.numProperties;i++){
      var eff = fx.property(i); if (!eff) continue;
      if (eff.matchName === "ADBE Fill"){
        for (var j=1;j<=eff.numProperties;j++){
          var p = eff.property(j); if (!p) continue;
          try{
            if (p.propertyValueType === PropertyValueType.COLOR) p.setValue(color01);
            if (p.matchName === "ADBE Fill-0006") p.setValue(100);
            if (p.matchName === "ADBE Fill-0007") p.setValue(0);
          }catch(e){}
        }
      }
      if (eff.matchName === "ADBE Tint"){
        for (var k=1;k<=eff.numProperties;k++){
          var t = eff.property(k); if (!t) continue;
          try{
            if (t.propertyValueType === PropertyValueType.COLOR) t.setValue(color01);
            if (t.propertyValueType === PropertyValueType.ONE_D && /Amount/i.test(t.name)) t.setValue(100);
          }catch(e){}
        }
      }
    }
  }
  function _syncLayerStyles(layer, color01){
    var ls = layer.property("ADBE Layer Styles") || layer.property("Layer Styles");
    if (!ls) return;
    var co = ls.property("ADBE Color Overlay") || ls.property("Color Overlay");
    if (co){
      var coColor   = co.property("ADBE Color Overlay-0002") || co.property("Color");
      var coOpacity = co.property("ADBE Color Overlay-0003") || co.property("Opacity");
      var coEnabled = co.property("ADBE Color Overlay-0001") || co.property("enabled");
      try{ if (coEnabled) coEnabled.setValue(true); if (coColor) coColor.setValue(color01); if (coOpacity) coOpacity.setValue(100); }catch(e){}
    }
    var st = ls.property("ADBE Stroke") || ls.property("Stroke");
    if (st){
      var sColor = st.property("ADBE Stroke-0003") || st.property("Color");
      try{ if (sColor) sColor.setValue(color01); }catch(e){}
    }
  }

  // TEXT fill+stroke
  function setTextFillAndStroke(layer, color01, strokeW){
    if (!layer) return false;
    var st = layer.property("Source Text"); if (!st) return false;
    var c = _safe01(color01);
    try{
      _killTextColorAnimators(layer);
      var td = st.value;
      td.applyFill   = true; td.fillColor   = c;
      td.applyStroke = true; td.strokeColor = c;
      if (strokeW !== null && strokeW !== undefined) td.strokeWidth = strokeW;
      try{ td.strokeOverFill = false; }catch(e){}
      st.setValue(td);
      _syncEffects(layer, c);
      _syncLayerStyles(layer, c);
      return true;
    }catch(e){ $.writeln("setTextFillAndStroke ERR: "+e); return false; }
  }

  // SHAPES fill+stroke (recursive)
  function setShapeFillAndStroke(layer, color01, strokeW){
    if (!layer) return 0;
    var c = _safe01(color01);
    var contents = layer.property("ADBE Root Vectors Group");
    if (!contents) return 0;
    function walk(group){
      var hits=0;
      for (var i=1;i<=group.numProperties;i++){
        var p = group.property(i);
        if (!p) continue;
        var mn = p.matchName;
        if (mn === "ADBE Vector Group" || mn === "ADBE Vector Filter" || mn === "ADBE Vector Shape - Group"){
          hits += walk(p);
        } else if (mn === "ADBE Vector Graphic - Fill"){
          var fc = p.property("ADBE Vector Fill Color");
          var fo = p.property("ADBE Vector Fill Opacity");
          try{ if (fc) fc.setValue(c); if (fo) fo.setValue(100); hits++; }catch(e){}
        } else if (mn === "ADBE Vector Graphic - Stroke"){
          var sc = p.property("ADBE Vector Stroke Color");
          var so = p.property("ADBE Vector Stroke Opacity");
          var sw = p.property("ADBE Vector Stroke Width");
          try{
            if (sc) sc.setValue(c);
            if (so) so.setValue(100);
            if (sw && strokeW!==null && strokeW!==undefined) sw.setValue(strokeW);
            hits++;
          }catch(e){}
        }
      }
      return hits;
    }
    return walk(contents);
  }

  // Deep updater by NAME: Text + Shape; recurses precomps
  function setFillAndStrokeOnNamedLayersDeep(compItem, namesArr, color01, strokeW, depth){
    depth = depth || 0; if (depth>12 || !compItem) return 0;
    var targets = {}; for (var n=0;n<namesArr.length;n++){ targets[_normLoose(namesArr[n])] = true; }
    var hits=0;
    for (var i=1;i<=compItem.numLayers;i++){
      var L = compItem.layer(i);
      try{
        var lname = _normLoose(L.name);
        if (targets[lname]){
          if (L.matchName === "ADBE TextLayer"){
            if (setTextFillAndStroke(L, color01, strokeW)) hits++;
          } else if (L.matchName === "ADBE Vector Layer"){
            hits += setShapeFillAndStroke(L, color01, strokeW);
          } else {
            _syncEffects(L, _safe01(color01));
            _syncLayerStyles(L, _safe01(color01));
          }
        }
        if (L.source && (L.source instanceof CompItem)){
          hits += setFillAndStrokeOnNamedLayersDeep(L.source, namesArr, color01, strokeW, depth+1);
        }
      }catch(e){}
    }
    return hits;
  }

  function setTeamNameTextAndStroke(compItem, nameStr, strokeColor01, strokeW){
    var lyr = GL.getLayer(compItem, LAYER_TEAMNAME);
    if (!lyr){
      var want=_normLoose(LAYER_TEAMNAME);
      for (var i=1;i<=compItem.numLayers;i++){ try{ if (_normLoose(compItem.layer(i).name)===want){ lyr = compItem.layer(i); break; } }catch(e){} }
    }
    if (!lyr) return 0;
    GL.setTextContent(lyr, nameStr||"");
    var st = lyr.property("Source Text"); if (!st) return 0;
    try{
      var td = st.value;
      td.applyStroke = true;
      td.strokeColor = _safe01(strokeColor01);
      if (strokeW!==null && strokeW!==undefined) td.strokeWidth = strokeW;
      st.setValue(td);
      return 1;
    }catch(e){ return 0; }
  }

  // Reuse-or-import then replace footage by layer name (deep)
  function replaceLayerFootageByNameDeep(compItem, targetName, filePath, depth){
    depth = depth||0; if (!compItem || !filePath) return 0;
    var want = _normLoose(targetName);
    var f = new File(filePath); if (!f.exists) { $.writeln("HomePlate file not found: "+filePath); return 0; }

    function findFootageByPath(pathFs){
      for (var i=1;i<=app.project.numItems;i++){
        var it = app.project.item(i);
        try { if (it instanceof FootageItem && it.file && it.file.fsName === pathFs) return it; } catch(e){}
      }
      return null;
    }

    var footage = findFootageByPath(f.fsName);
    if (!footage){
      var io = new ImportOptions(f);
      if (!io.canImportAs(ImportAsType.FOOTAGE)) { $.writeln("Cannot import as footage: "+filePath); return 0; }
      io.importAs = ImportAsType.FOOTAGE;
      footage = app.project.importFile(io);
    }

    var hits = 0;
    for (var i=1;i<=compItem.numLayers;i++){
      var L = compItem.layer(i);
      try{
        var nm = _normLoose(L.name);
        if (nm === want && L.source){
          try { L.replaceSource(footage, false); hits++; } catch(e){}
        }
        if (L.source && (L.source instanceof CompItem)){
          hits += replaceLayerFootageByNameDeep(L.source, targetName, filePath, depth+1);
        }
      }catch(e){}
    }
    return hits;
  }

  // --- per run setup ---
  $.writeln("\n========================================");
  $.writeln("Processing " + todo.length + " teams for league: " + LEAGUE);
  $.writeln("Animation tag: " + ANIM_NAME);
  $.writeln("========================================");

  // ONE plate swap per run
  if (HOMEPLATE_PATH && LAYER_HOMEPLATE){
    var swapped = replaceLayerFootageByNameDeep(comp, LAYER_HOMEPLATE, HOMEPLATE_PATH, 0);
    $.writeln("HomePlate replacements: " + swapped + "  (" + HOMEPLATE_PATH + ")");
  }

  // --- team loop ---
  for (var i=0;i<todo.length;i++){
    var t = todo[i];
    var smart = GL.computeSmartColors( GL.safeColor(t.primary), GL.safeColor(t.secondary) );
    var P = smart.primary, S = smart.secondary;

    // HOMERUN
    var hr = GL.getLayer(comp, LAYER_HOMERUN);
    setTextFillAndStroke(hr, P, STROKE_W);

    // INNER HOME/RUN in precomp
    var hrPre = GL.getLayer(comp, LAYER_HR_PRECOMP);
    if (hrPre && hrPre.source && (hrPre.source instanceof CompItem)){
      setFillAndStrokeOnNamedLayersDeep(hrPre.source, INNER_STROKE_NAMES, P, STROKE_W, 0);
    }

    // Logo
    GL.replaceLogo(comp, LAYER_LOGO, t.league, t.abbr, logoOpts);

    // Team name stroke
    setTeamNameTextAndStroke(comp, t.name, S, STROKE_W);

    // Pixel snap
    GL.snapCompTextForPixelArt(comp);

    if (PURGE && app.purge){ try{ app.purge(PurgeTarget.ALL_CACHES); }catch(e){} }

    // Render
    if (!NO_RENDER){
      var lc    = GL.leagueAndConfForPath(t.league, t.conference);
      var paths = GL.outPaths(rootOut, PATH_TPL, lc.base, t.abbr, ANIM_NAME, lc.conf, t.espn_team_id);
      $.writeln("  Output: " + paths.file.fsName);
      GL.rqRenderTo(comp, RS_TPL, OM_TPL, paths.file);
    }

    // Clean up logo from project after render
    var removed = GL.removeLogoByAbbr(t.abbr);
    if (removed > 0) $.writeln("Removed " + removed + " logo(s) for " + t.abbr);
  }

  app.endUndoGroup();
  if (app.endSuppressDialogs){ try{ app.endSuppressDialogs(); }catch(e){} }

  if (QUIT_APP) app.quit();
})();
